{Simple Int 13h-based Floppy Drive PFS driver}

{$A+,B-,D+,E-,F-,G+,I+,L+,N-,O-,P-,Q-,R-,S+,T-,V-,X+,Y+}
Unit BIOSFloppyDriver;

Interface

Uses
   PFSTypes,PFSDrivers;

(**************************************************************************)

Procedure PFS_LoadBIOSFloppyDriver;

(**************************************************************************)

Implementation

(**************************************************************************)

{Internal types}

Const
   NumRWRetries=3;  {Number of retries for read/write operations}

   {FD drive types}
   Floppy_dt360 =$01;
   Floppy_dt1200=$02;
   Floppy_dt720 =$03;
   Floppy_dt1440=$04;
   Floppy_dt2880=$06; {can also be 5}

Type
   PFloppyDiskInfo=^TFloppyDiskInfo;
   TFloppyDiskInfo=Record
                {00}  Size:TCHS; {Number of cylinders, heads, and sectors}
                {03}  DriveType:Byte;
                {04}
                   End;

Var
   NextDriveToDo:Word;

(**************************************************************************)

{API functions for driver}

(**************************************************************************)

{InitDriver - nothing to really do here}

Function Floppy_InitDriver:Word; Far;

Begin
   NextDriveToDo:=0;
   Floppy_InitDriver:=PFS_ecOk;
End;

(**************************************************************************)

{DoneDriver - nothing to really do here either}

Function Floppy_DoneDriver:Word; Far;

Begin
   Floppy_DoneDriver:=PFS_ecOk;
End;

(**************************************************************************)

{Internal asm routines}

Procedure FDC_Reset; Assembler;

Asm
         xor  ah,ah
         xor  dl,dl
         int  15h
End;

Function FDC_Status(Drive:Byte):Byte; Assembler;

Asm
         mov  ah,1
         mov  dl,[Drive]
         int  15h
         mov  al,ah
End;

(**************************************************************************)

{Resets the status and gets info for the drive}

Procedure Floppy_ResetDriveInfo(var Drive:TPFS_Drive);

Const
   InitialStatus=PFS_dsPresent or PFS_dsReadable or PFS_dsWritable or PFS_dsDiskInDrive;

Var
   Status:Byte;

Begin
   FDC_Reset;
   Status:=FDC_Status(Drive.DriverDriveNumber);
   Drive.Status:=InitialStatus;
   Case Status of
      {Disk write-protected}
      $03 : Drive.Status:=Drive.Status and not(PFS_dsWritable);
      {Disk changed, so turn off all I/O}
      $06 : Drive.Status:=Drive.Status and not
                           (PFS_dsReadable or PFS_dsWritable);
   End;
End;

(**************************************************************************)

{Convert Logical Sector to CHS}

Procedure ConvertToCHS(LogicalSector:TPFS_LogicalSectorNumber;
                       Size:TCHS; var CHS:TCHS);

Begin
   With CHS Do
      Begin
         Sector:=(LogicalSector.Lo mod Size.Sector)+1;
         LogicalSector.Lo:=LogicalSector.Lo div Size.Sector;
         Head:=LogicalSector.Lo mod (Size.Head+1);
         Cylinder:=LogicalSector.Lo div (Size.Head+1);
      End;
End;

(**************************************************************************)

{InitDrive}

Function Floppy_InitDrive(var Drive:TPFS_Drive):Word; Far;

Begin
   Drive.DriverDriveNumber:=NextDriveToDo;
   Inc(NextDriveToDo);
   New(PFloppyDiskInfo(Drive.DriverInfo));
   Asm
         les  di,[Drive]                {ES:DI -> Drive}
         mov  dl,[es:di+$1E]            {DL = Drive.DriverDriveNumber}
         les  di,[es:di+$26]            {ES:DI -> Drive.DriverInfo^}
         push es
         mov  ah,08h                    {Funciton 08h}
         push di                        {save ES and DI}
         int  13h                       {call FDC BIOS}
         pop  di
         pop  es                        {restore ES:DI}
         mov  [es:di],dh                {Size.Head}
         mov  [es:di+$01],cl            {Size.Sector}
         mov  [es:di+$02],ch            {Size.Cylinder}
         mov  [es:di+$03],bl            {DriveType}
(* Below useful for HD's which use bits 6-7 of cylinder.  FD's don't
   however, so I'll use the shorter version instead.
         mov  al,dh
         and  al,3Fh                    {AL = max head}
         inc  al
         mul  cl                        {AX = num heads * numsectors}
         mov  bl,ch
         mov  bh,dh
         shr  bh,6                      {BX = max cylinder}
         mul  bx                        {DX:AX = total num sectors}
*)
         mov  al,cl
         inc  dh
         xor  cl,cl
         mul  dh
         xchg cl,ch
         inc  cx
         mul  cx
         les  di,[Drive]
 db $66; xor  bx,bx                     {EBX = 0}
         mov  [es:di+$14],ax
         mov  [es:di+$16],dx            {set TotalSectors.Lo = DX:AX}
 db $66; mov  [es:di+$18],bx            {set TotalSectors.Hi = 0}
   End;
   Floppy_ResetDriveInfo(Drive);
   Floppy_InitDrive:=PFS_ecOK;
End;

(**************************************************************************)

{DoneDrive}

Function Floppy_DoneDrive(var Drive:TPFS_Drive):Word; Far;

Begin
   Dispose(PFloppyDiskInfo(Drive.DriverInfo));
   Drive.Status:=0;
   Floppy_DoneDrive:=PFS_ecOk;
End;

(**************************************************************************)

{DetectHardware}

Function Floppy_DetectHardware(var NumDrives:Word):Word; Far; Assembler;

Asm
         mov  ah,08h
         xor  dl,dl
         int  13h
         les  di,[NumDrives]
         xor  dh,dh
         mov  [es:di],dx
         xor  ax,ax                     {mov ax,PFS_ecOk}
End;

(**************************************************************************)

Function Floppy_ReadSector(var Drive:TPFS_Drive; Sector:PPFS_Sector;
                           SectorNumber:TPFS_LogicalSectorNumber):Word; Far;

Var
   PhysicalSector:TCHS;
   SaveCX:Word;

Begin
   ConvertToCHS(SectorNumber,PFloppyDiskInfo(Drive.DriverInfo)^.Size,
               PhysicalSector);
   Asm
         les  di,[Drive]
         mov  dl,[es:di+$1E]
         mov  ch,[PhysicalSector.Cylinder]
         mov  cl,[PhysicalSector.Sector]
         mov  dh,[PhysicalSector.Head]
         les  bx,[Sector]
         mov  [SaveCX],cx
         mov  cx,NumRWRetries
@@Top:   mov  ax,0201h
         xchg [SaveCX],cx
         int  13h
         jnc  @@Ok
         xchg [SaveCX],cx
         xor  ah,ah
         int  13h
         loop @@Top
         mov  [@Result],PFS_ecFail
         jmp  @@End
@@Ok:    mov  [@Result],PFS_ecOk
@@End:
   End;
End;


(**************************************************************************)

Function Floppy_WriteSector(var Drive:TPFS_Drive; Sector:PPFS_Sector;
                            SectorNumber:TPFS_LogicalSectorNumber):Word; Far;

Var
   PhysicalSector:TCHS;
   SaveCX:Word;

Begin
   ConvertToCHS(SectorNumber,PFloppyDiskInfo(Drive.DriverInfo)^.Size,
               PhysicalSector);
   Asm
         les  di,[Drive]
         mov  dl,[es:di+$1E]
         mov  ch,[PhysicalSector.Cylinder]
         mov  cl,[PhysicalSector.Sector]
         mov  dh,[PhysicalSector.Head]
         les  bx,[Sector]
         mov  [SaveCX],cx
         mov  cx,NumRWRetries
@@Top:   mov  ax,0301h
         xchg [SaveCX],cx
         int  13h
         jnc  @@Ok
         xchg [SaveCX],cx
         xor  ah,ah
         int  13h
         loop @@Top
         mov  [@Result],PFS_ecFail
         jmp  @@End
@@Ok:    mov  [@Result],PFS_ecOk
@@End:
   End;
End;

(**************************************************************************)

Function Floppy_Format(var Drive:TPFS_Drive; StartSector,EndSector:
                       TPFS_LogicalSectorNumber):Word; Far;

Const
   Sector_512=$02;

Type
   TAFBE=Record {Address Field Buffer Entry}
            Track:Byte;
            Head:Byte;
            Sector:Byte;
            SectorSize:Byte;
         End;
   TAFB=Array[1..63] of TAFBE;  {Max of 63 sectors/track}
   PAFB=^TAFB;

Var
   Start,Finish,Current,Size:TCHS;
   Ok:Boolean;
   Loop:Byte;
   AFB:PAFB;
   DriveNumber:Byte;

Begin
   If (StartSector.Lo>=Drive.TotalSectors.Lo) or
      (EndSector.Lo>=Drive.TotalSectors.Lo) or
      (StartSector.Lo>EndSector.Lo) Then
         Begin
            Floppy_Format:=PFS_ecBadSectorNumber;
            Exit;
         End;
   Size:=PFloppyDiskInfo(Drive.DriverInfo)^.Size;
   ConvertToCHS(StartSector,Size,Start);
   ConvertToCHS(EndSector,Size,Finish);
   If (Start.Sector<>1) or (Finish.Sector<>Size.Sector) Then
      Begin
         Floppy_Format:=PFS_ecBadSectorNumber;
         Exit;
      End;
   DriveNumber:=Drive.DriverDriveNumber;
   Asm
         mov  ch,[Size.Cylinder]
         mov  cl,[Size.Sector]
         les  di,[Drive]
         mov  dl,[DriveNumber]
         mov  ah,18h                    {set Media Type for format}
         int  13h
         mov  [Ok],True
         or   ah,ah
         jz   @@Ok
         mov  [Ok],False
         mov  [@Result],PFS_ecFail
         cmp  ah,80h
         jnz  @@Ok
         mov  [@Result],PFS_ecNoDiskInDrive
   @@Ok:
   End;
   If Not(Ok) Then
      Exit;
   GetMem(AFB,SizeOf(TAFBE)*Size.Sector);
   Current:=Start;
   Repeat
      For Loop:=1 to Size.Sector Do
         With AFB^[Loop] Do
            Begin
               Track:=Current.Cylinder;
               Head:=Current.Head;
               Sector:=Loop;
               SectorSize:=Sector_512;
            End;
      Asm
            mov   al,[Size.Sector]
            mov   ch,[Current.Cylinder]
            mov   dh,[Current.Head]
            mov   dl,[DriveNumber]
            mov   ah,05h
            les   bx,[AFB]
            int   13h
            mov   [Ok],True
            jnc   @@Ok
            mov   [Ok],False
      @@Ok:
      End;
      If Not(Ok) Then
         Begin
            Floppy_Format:=PFS_ecFail;
            Exit;
         End;
      Inc(Current.Head);
      If(Current.Head>Size.Head) Then
         Begin
            Current.Head:=0;
            Inc(Current.Cylinder);
         End;
   Until (Current.Cylinder>Finish.Cylinder) or
         ((Current.Cylinder=Finish.Cylinder) and (Current.Head>Finish.Head));
   FreeMem(AFB,SizeOf(TAFBE)*Size.Sector);
   Floppy_Format:=PFS_ecOk;
End;


(**************************************************************************)

{Driver declaration}

Const
   PFS_BIOSFloppyDriverAPI:TPFS_DriverFunctions=
      (InitDriver: Floppy_InitDriver;
       DoneDriver: Floppy_DoneDriver;
       InitDrive: Floppy_InitDrive;
       DoneDrive: Floppy_DoneDrive;
       DetectHardware: Floppy_DetectHardware;
       ReadSector: Floppy_ReadSector;
       WriteSector: Floppy_WriteSector;
       Format: Floppy_Format);
   PFS_BIOSFloppyDriver:TPFS_Driver=
      (Name: 'Floppy - Int 13';
       Version: (Major: 1; Minor: 0);
       API: @PFS_BIOSFloppyDriverAPI;
       NumDrives: 0);

(**************************************************************************)

Procedure PFS_LoadBIOSFloppyDriver;

Begin
   PFS_RegisterDriver(@PFS_BIOSFloppyDriver);
End;

(**************************************************************************)

End.